import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  PanResponder,
  Dimensions,
} from 'react-native';
import { useTheme } from '../theme/ThemeContext';
import { usePlayerStore } from '../store/playerStore';
import { usePlayerControls, formatTime } from '../hooks/usePlayer';

interface ProgressBarProps {
  showTime?: boolean;
}

export function ProgressBar({ showTime = true }: ProgressBarProps) {
  const { colors } = useTheme();
  const { position, duration } = usePlayerStore();
  const { seekTo } = usePlayerControls();
  const [isSeeking, setIsSeeking] = useState(false);
  const [seekPosition, setSeekPosition] = useState(0);

  const progress = duration > 0 ? (position / duration) * 100 : 0;
  const displayProgress = isSeeking ? seekPosition : progress;

  const panResponder = PanResponder.create({
    onStartShouldSetPanResponder: () => true,
    onMoveShouldSetPanResponder: () => true,
    onPanResponderGrant: (evt) => {
      setIsSeeking(true);
      const { locationX } = evt.nativeEvent;
      const width = Dimensions.get('window').width - 48; // Account for padding
      const newProgress = Math.max(0, Math.min(100, (locationX / width) * 100));
      setSeekPosition(newProgress);
    },
    onPanResponderMove: (evt) => {
      const { locationX } = evt.nativeEvent;
      const width = Dimensions.get('window').width - 48;
      const newProgress = Math.max(0, Math.min(100, (locationX / width) * 100));
      setSeekPosition(newProgress);
    },
    onPanResponderRelease: () => {
      setIsSeeking(false);
      const newPosition = (seekPosition / 100) * duration;
      seekTo(newPosition);
    },
  });

  const displayTime = isSeeking ? (seekPosition / 100) * duration : position;

  return (
    <View style={styles.container}>
      <View
        style={[styles.track, { backgroundColor: colors.border }]}
        {...panResponder.panHandlers}
      >
        <View
          style={[
            styles.fill,
            { width: `${displayProgress}%`, backgroundColor: colors.primary },
          ]}
        />
        <View
          style={[
            styles.thumb,
            {
              left: `${displayProgress}%`,
              backgroundColor: colors.primary,
            },
          ]}
        />
      </View>

      {showTime && (
        <View style={styles.timeContainer}>
          <Text style={[styles.time, { color: colors.textSecondary }]}>
            {formatTime(displayTime)}
          </Text>
          <Text style={[styles.time, { color: colors.textSecondary }]}>
            {formatTime(duration)}
          </Text>
        </View>
      )}
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    width: '100%',
    paddingHorizontal: 24,
  },
  track: {
    height: 4,
    borderRadius: 2,
    position: 'relative',
  },
  fill: {
    height: '100%',
    borderRadius: 2,
  },
  thumb: {
    position: 'absolute',
    top: -6,
    width: 16,
    height: 16,
    borderRadius: 8,
    marginLeft: -8,
  },
  timeContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: 8,
  },
  time: {
    fontSize: 12,
  },
});
