import React from 'react';
import { View, Text, StyleSheet } from 'react-native';
import { createBottomTabNavigator } from '@react-navigation/bottom-tabs';
import { useTheme } from '../theme/ThemeContext';
import { LibraryScreen } from '../screens/LibraryScreen';
import { SearchScreen } from '../screens/SearchScreen';
import { SettingsScreen } from '../screens/SettingsScreen';
import { MiniPlayer } from '../components/MiniPlayer';
import { usePlayerStore } from '../store/playerStore';

const Tab = createBottomTabNavigator();

interface TabIconProps {
  focused: boolean;
  color: string;
  label: string;
  icon: string;
}

function TabIcon({ focused, color, label, icon }: TabIconProps) {
  return (
    <View style={styles.tabIconContainer}>
      <Text style={[styles.tabIcon, { color }]}>{icon}</Text>
      <Text style={[styles.tabLabel, { color }]}>{label}</Text>
    </View>
  );
}

interface TabNavigatorProps {
  onMiniPlayerPress: () => void;
}

export function TabNavigator({ onMiniPlayerPress }: TabNavigatorProps) {
  const { colors } = useTheme();
  const currentTrack = usePlayerStore((s) => s.currentTrack);

  return (
    <View style={styles.container}>
      <Tab.Navigator
        screenOptions={{
          headerShown: false,
          tabBarStyle: {
            backgroundColor: colors.tabBar,
            borderTopColor: colors.tabBarBorder,
            borderTopWidth: 1,
            height: 60 + (currentTrack ? 64 : 0), // Extra space for mini player
            paddingBottom: 8,
            paddingTop: currentTrack ? 64 : 8, // Push tabs down when mini player visible
          },
          tabBarActiveTintColor: colors.primary,
          tabBarInactiveTintColor: colors.textMuted,
          tabBarShowLabel: false,
        }}
      >
        <Tab.Screen
          name="Library"
          component={LibraryScreen}
          options={{
            tabBarIcon: ({ focused, color }) => (
              <TabIcon
                focused={focused}
                color={color}
                label="Library"
                icon={'\uD83C\uDFB5'} // Musical note
              />
            ),
          }}
        />
        <Tab.Screen
          name="Add"
          component={SearchScreen}
          options={{
            tabBarIcon: ({ focused, color }) => (
              <TabIcon
                focused={focused}
                color={color}
                label="Add"
                icon="+"
              />
            ),
          }}
        />
        <Tab.Screen
          name="Settings"
          component={SettingsScreen}
          options={{
            tabBarIcon: ({ focused, color }) => (
              <TabIcon
                focused={focused}
                color={color}
                label="Settings"
                icon={'\u2699'} // Gear
              />
            ),
          }}
        />
      </Tab.Navigator>

      {/* Mini Player overlay */}
      {currentTrack && (
        <View style={styles.miniPlayerContainer}>
          <MiniPlayer onPress={onMiniPlayerPress} />
        </View>
      )}
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  tabIconContainer: {
    alignItems: 'center',
    justifyContent: 'center',
  },
  tabIcon: {
    fontSize: 22,
    marginBottom: 2,
  },
  tabLabel: {
    fontSize: 10,
    fontWeight: '500',
  },
  miniPlayerContainer: {
    position: 'absolute',
    bottom: 52, // Just above tab bar
    left: 0,
    right: 0,
  },
});
