# Vinyl - Web-Based Audio Player

## Project Overview
A self-hosted, multi-user audio player web app. Users register, upload audio files to the server, create playlists, and stream from any device. Works on iOS without app store restrictions.

**URL:** `https://rosy.shitchell.com/vinyl/`
**API:** `https://rosy.shitchell.com/vinyl-api/`
**Location:** `~/projects/vinyl-web` (development), `/var/www/rosy.shitchell.com/srv/vinyl-*` (deployed)
**Audio Storage:** Configurable via `UPLOAD_DIR` env var (default: `/data/vinyl/uploads/`)

---

## Tech Stack

| Layer | Technology |
|-------|------------|
| **Frontend** | Vanilla JS + HTML/CSS (PWA-capable) |
| **Backend** | Node.js + Express |
| **ORM** | Sequelize (supports PostgreSQL, SQLite, MariaDB, MySQL) |
| **Database** | MariaDB (your existing server) |
| **Auth** | Express sessions + bcrypt |
| **File Storage** | Local filesystem (server) |
| **Audio** | HTML5 Audio API |

---

## Database Schema

### users
```sql
id            INT PRIMARY KEY AUTO_INCREMENT
username      VARCHAR(50) UNIQUE NOT NULL
password_hash VARCHAR(255) NOT NULL
created_at    DATETIME DEFAULT CURRENT_TIMESTAMP
```

### tracks
```sql
id            INT PRIMARY KEY AUTO_INCREMENT
user_id       INT NOT NULL (FK → users)
title         VARCHAR(255) NOT NULL
artist        VARCHAR(255)
album         VARCHAR(255)
duration      INT (seconds)
filename      VARCHAR(255) NOT NULL (server path)
original_name VARCHAR(255) (uploaded filename)
size_bytes    BIGINT
mime_type     VARCHAR(50)
created_at    DATETIME DEFAULT CURRENT_TIMESTAMP
```

### playlists
```sql
id            INT PRIMARY KEY AUTO_INCREMENT
user_id       INT NOT NULL (FK → users)
name          VARCHAR(100) NOT NULL
is_public     BOOLEAN DEFAULT FALSE
share_code    VARCHAR(20) UNIQUE (for sharing private playlists)
created_at    DATETIME DEFAULT CURRENT_TIMESTAMP
updated_at    DATETIME
```

### playlist_tracks
```sql
playlist_id   INT (FK → playlists)
track_id      INT (FK → tracks)
position      INT NOT NULL
PRIMARY KEY (playlist_id, track_id)
```

---

## API Endpoints

### Auth
| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/auth/register` | Create account |
| POST | `/api/auth/login` | Login, create session |
| POST | `/api/auth/logout` | Destroy session |
| GET | `/api/auth/me` | Get current user |

### Tracks
| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/api/tracks` | List user's tracks |
| POST | `/api/tracks` | Upload new track(s) |
| GET | `/api/tracks/:id` | Get track metadata |
| DELETE | `/api/tracks/:id` | Delete track |
| GET | `/api/tracks/:id/stream` | Stream audio file |
| GET | `/api/tracks/:id/download` | Download audio file |

### Playlists
| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/api/playlists` | List user's playlists |
| POST | `/api/playlists` | Create playlist |
| GET | `/api/playlists/:id` | Get playlist with tracks |
| PUT | `/api/playlists/:id` | Update playlist (name, track order) |
| DELETE | `/api/playlists/:id` | Delete playlist |
| POST | `/api/playlists/:id/tracks` | Add track to playlist |
| DELETE | `/api/playlists/:id/tracks/:trackId` | Remove track from playlist |
| POST | `/api/playlists/:id/share` | Generate share link |
| GET | `/api/playlists/shared/:shareCode` | View shared playlist (no auth required) |

---

## Project Structure

```
vinyl-web/
├── server/
│   ├── index.js              # Express entry point
│   ├── config/
│   │   └── database.js       # Sequelize config
│   ├── models/
│   │   ├── index.js          # Model associations
│   │   ├── User.js
│   │   ├── Track.js
│   │   ├── Playlist.js
│   │   └── PlaylistTrack.js
│   ├── routes/
│   │   ├── auth.js
│   │   ├── tracks.js
│   │   └── playlists.js
│   ├── middleware/
│   │   └── auth.js           # Session validation
│   └── uploads/              # Audio file storage
├── client/
│   ├── index.html
│   ├── css/
│   │   └── style.css
│   ├── js/
│   │   ├── app.js            # Main app logic
│   │   ├── api.js            # API client
│   │   ├── player.js         # Audio player logic
│   │   ├── ui.js             # DOM manipulation
│   │   └── auth.js           # Login/register
│   └── manifest.json         # PWA manifest
├── package.json
└── .env.example
```

---

## Implementation Phases

### Phase 1: Project Setup
- [ ] Create `vinyl-web` directory structure
- [ ] Initialize npm, install dependencies
- [ ] Set up Sequelize with MariaDB connection
- [ ] Create `.env` with database credentials

**Dependencies:**
```json
{
  "express": "^4.18",
  "sequelize": "^6.35",
  "mysql2": "^3.6",
  "bcrypt": "^5.1",
  "express-session": "^1.17",
  "multer": "^1.4",
  "dotenv": "^16.3",
  "cors": "^2.8"
}
```

### Phase 2: Database & Models
- [ ] Define Sequelize models (User, Track, Playlist, PlaylistTrack)
- [ ] Set up model associations
- [ ] Run migrations / sync models
- [ ] Test database connection

### Phase 3: Auth System
- [ ] Implement `/api/auth/register` with bcrypt password hashing
- [ ] Implement `/api/auth/login` with session creation
- [ ] Implement `/api/auth/logout`
- [ ] Create auth middleware for protected routes
- [ ] Build login/register UI

### Phase 4: Track Upload & Streaming
- [ ] Set up Multer for file uploads
- [ ] Implement `/api/tracks` POST (upload)
- [ ] Implement `/api/tracks/:id/stream` (streaming with range requests)
- [ ] Implement `/api/tracks` GET (list)
- [ ] Implement `/api/tracks/:id` DELETE
- [ ] Extract audio metadata (duration) if possible

### Phase 5: Frontend Player
- [ ] Build HTML structure (library, player bar, now playing)
- [ ] Style with CSS (dark theme, mobile-friendly)
- [ ] Implement audio player with HTML5 Audio API
- [ ] Track list display
- [ ] Play/pause, next/previous, seek bar
- [ ] Queue management
- [ ] Shuffle/repeat modes

### Phase 6: Playlists & Sharing
- [ ] Implement playlist CRUD endpoints
- [ ] Build playlist UI (create, view, edit)
- [ ] Drag-and-drop reordering (stretch goal)
- [ ] Implement share link generation (unique share_code)
- [ ] Public shared playlist view (no login required)
- [ ] Copy share link to clipboard

### Phase 7: Download & Polish
- [ ] Implement download endpoint with proper headers
- [ ] Add File System Access API for Chrome (save to folder)
- [ ] Fallback to standard download for iOS/other browsers
- [ ] PWA manifest for "Add to Home Screen"
- [ ] Service Worker for basic offline support (app shell caching)

### Phase 8: Deployment
- [ ] Copy server to `/var/www/rosy.shitchell.com/srv/vinyl-api/`
- [ ] Copy client to `/var/www/rosy.shitchell.com/srv/vinyl/`
- [ ] Add nginx location blocks for `/vinyl/` and `/vinyl-api/`
- [ ] Set up systemd service or PM2 for the Node server
- [ ] Test end-to-end

---

## Nginx Config (to add)

```nginx
# Vinyl Web App
location /vinyl/ {
    alias /var/www/rosy.shitchell.com/srv/vinyl/;
    index index.html;
    try_files $uri $uri/ /vinyl/index.html;
}

# Vinyl API
location /vinyl-api/ {
    proxy_pass http://127.0.0.1:8085/;
    proxy_set_header Host $host;
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    client_max_body_size 100M;  # Allow large audio uploads
}
```

---

## Security Considerations

- Passwords hashed with bcrypt (cost factor 10+)
- Sessions stored server-side (not JWT in localStorage)
- File uploads validated for audio MIME types
- User can only access their own tracks/playlists
- HTTPS enforced (already configured)

---

## Environment Variables (.env)

```bash
# Database
DB_DIALECT=mariadb          # or: postgres, sqlite, mysql
DB_HOST=localhost
DB_PORT=3306
DB_NAME=vinyl
DB_USER=rosy
DB_PASS=fakan

# Session
SESSION_SECRET=change-this-to-random-string

# Storage
UPLOAD_DIR=/data/vinyl/uploads   # Configurable path for audio files

# Server
PORT=8085
```

---

## Future Enhancements (not in initial scope)

- Audio waveform visualization
- Album art extraction/upload
- Storage quotas per user
- Mobile app using this as backend
- Collaborative playlists (multiple users can edit)
