import React from 'react';
import { View, Text, TouchableOpacity, StyleSheet } from 'react-native';
import { useTheme } from '../theme/ThemeContext';
import { usePlayerStore } from '../store/playerStore';
import { usePlayerControls } from '../hooks/usePlayer';

interface PlayerControlsProps {
  size?: 'small' | 'large';
}

export function PlayerControls({ size = 'large' }: PlayerControlsProps) {
  const { colors } = useTheme();
  const { isPlaying, shuffle, repeatMode } = usePlayerStore();
  const { play, pause, skipToNext, skipToPrevious, toggleShuffle, cycleRepeatMode } =
    usePlayerControls();

  const isLarge = size === 'large';
  const iconSize = isLarge ? 32 : 24;
  const playIconSize = isLarge ? 48 : 32;

  const getRepeatIcon = () => {
    switch (repeatMode) {
      case 'track':
        return '\u{1F502}'; // Repeat one emoji
      case 'queue':
        return '\u{1F501}'; // Repeat emoji
      default:
        return '\u{1F501}'; // Repeat emoji (dimmed)
    }
  };

  return (
    <View style={styles.container}>
      {/* Shuffle */}
      <TouchableOpacity
        onPress={toggleShuffle}
        style={styles.sideButton}
      >
        <Text
          style={[
            styles.icon,
            { fontSize: iconSize - 8 },
            { color: shuffle ? colors.primary : colors.textMuted },
          ]}
        >
          {'\u{1F500}'} {/* Shuffle emoji */}
        </Text>
      </TouchableOpacity>

      {/* Previous */}
      <TouchableOpacity onPress={skipToPrevious} style={styles.button}>
        <Text style={[styles.icon, { fontSize: iconSize, color: colors.text }]}>
          {'\u23EE'} {/* Previous track */}
        </Text>
      </TouchableOpacity>

      {/* Play/Pause */}
      <TouchableOpacity
        onPress={isPlaying ? pause : play}
        style={[
          styles.playButton,
          { backgroundColor: colors.primary },
          isLarge && styles.playButtonLarge,
        ]}
      >
        <Text style={[styles.playIcon, { fontSize: playIconSize, color: '#fff' }]}>
          {isPlaying ? '\u275A\u275A' : '\u25B6'}
        </Text>
      </TouchableOpacity>

      {/* Next */}
      <TouchableOpacity onPress={skipToNext} style={styles.button}>
        <Text style={[styles.icon, { fontSize: iconSize, color: colors.text }]}>
          {'\u23ED'} {/* Next track */}
        </Text>
      </TouchableOpacity>

      {/* Repeat */}
      <TouchableOpacity
        onPress={cycleRepeatMode}
        style={styles.sideButton}
      >
        <Text
          style={[
            styles.icon,
            { fontSize: iconSize - 8 },
            { color: repeatMode !== 'off' ? colors.primary : colors.textMuted },
          ]}
        >
          {getRepeatIcon()}
        </Text>
      </TouchableOpacity>
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  button: {
    padding: 12,
    marginHorizontal: 8,
  },
  sideButton: {
    padding: 12,
    marginHorizontal: 4,
  },
  playButton: {
    width: 56,
    height: 56,
    borderRadius: 28,
    justifyContent: 'center',
    alignItems: 'center',
    marginHorizontal: 16,
  },
  playButtonLarge: {
    width: 72,
    height: 72,
    borderRadius: 36,
  },
  icon: {
    textAlign: 'center',
  },
  playIcon: {
    textAlign: 'center',
    marginLeft: 2, // Slight offset for play icon
  },
});
