#!/home/rosy/bin/mariadb

-- =====================================================
-- CTEs and CROSS JOIN: Comparing Against a Top Value
-- =====================================================

-- STEP 1: Let's start simple - get total deliveries per warehouse for 2023
SELECT
    warehouse_id,
    warehouse_name,
    SUM(total_shipments) AS total_deliveries
FROM daily_delivery_metrics
WHERE YEAR(report_date) = 2023
GROUP BY warehouse_id, warehouse_name
ORDER BY total_deliveries DESC
LIMIT 5;

-- STEP 2: Now, what if we just want THE TOP warehouse's number?
-- We could run a separate query...
SELECT SUM(total_shipments) AS top_deliveries
FROM daily_delivery_metrics
WHERE YEAR(report_date) = 2023
GROUP BY warehouse_id
ORDER BY top_deliveries DESC
LIMIT 1;

-- STEP 3: But we need BOTH in one query. Enter CTEs!
-- A CTE (Common Table Expression) is like a temporary table that only
-- exists for one query. Think of it as "let me calculate this first,
-- then use it below."

-- This CTE calculates totals, but doesn't display anything yet:
WITH totals AS (
    SELECT
        warehouse_id,
        warehouse_name,
        SUM(total_shipments) AS total_deliveries
    FROM daily_delivery_metrics
    WHERE YEAR(report_date) = 2023
    GROUP BY warehouse_id, warehouse_name
)
SELECT * FROM totals ORDER BY total_deliveries DESC LIMIT 5;

-- STEP 4: We can chain CTEs! Let's add a second one for the top value:
WITH totals AS (
    SELECT
        warehouse_id,
        warehouse_name,
        SUM(total_shipments) AS total_deliveries
    FROM daily_delivery_metrics
    WHERE YEAR(report_date) = 2023
    GROUP BY warehouse_id, warehouse_name
),
top_warehouse AS (
    -- This CTE uses the first CTE! It grabs just the top number.
    SELECT total_deliveries
    FROM totals
    ORDER BY total_deliveries DESC
    LIMIT 1
)
SELECT * FROM top_warehouse;

-- STEP 5: Now we need to COMBINE them. Enter CROSS JOIN!
--
-- A regular JOIN matches rows based on a condition (like customer_id = id).
-- A CROSS JOIN pairs EVERY row from table A with EVERY row from table B.
--
-- Since top_warehouse has just 1 row, CROSS JOIN adds that value to
-- every warehouse row - exactly what we need for comparison!

WITH totals AS (
    SELECT
        warehouse_id,
        warehouse_name,
        SUM(total_shipments) AS total_deliveries
    FROM daily_delivery_metrics
    WHERE YEAR(report_date) = 2023
    GROUP BY warehouse_id, warehouse_name
),
top_warehouse AS (
    SELECT total_deliveries AS top_value
    FROM totals
    ORDER BY total_deliveries DESC
    LIMIT 1
)
SELECT
    t.warehouse_name,
    t.total_deliveries,
    tw.top_value,
    '← same value on every row!' AS notice
FROM totals t
CROSS JOIN top_warehouse tw
ORDER BY t.total_deliveries DESC
LIMIT 5;

-- STEP 6: Finally, calculate the percentage difference!
WITH totals AS (
    SELECT
        warehouse_id,
        warehouse_name,
        SUM(total_shipments) AS total_deliveries
    FROM daily_delivery_metrics
    WHERE YEAR(report_date) = 2023
    GROUP BY warehouse_id, warehouse_name
),
top_warehouse AS (
    SELECT total_deliveries AS top_value
    FROM totals
    ORDER BY total_deliveries DESC
    LIMIT 1
)
SELECT
    t.warehouse_name,
    t.total_deliveries,
    tw.top_value AS top_deliveries,
    ROUND(
        ((t.total_deliveries - tw.top_value) / tw.top_value) * 100,
        1
    ) AS pct_diff_from_top
FROM totals t
CROSS JOIN top_warehouse tw
ORDER BY t.total_deliveries DESC;
