# Thru My Lens - Design Thought Process

## Introduction

This document explains the reasoning and thought process behind each design decision for Rosy's photography portfolio website. The goal was to create a space that feels personal, elegant, and lets the photographs be the star of the show.

---

## 1. Overall Vision

### The Problem
Photography portfolios often fall into two traps:
1. **Too busy** - Flashy templates that distract from the photos
2. **Too generic** - Stock templates that feel impersonal

### The Solution
Create a website that feels like walking into a quiet, well-lit gallery. The design should:
- Feel warm and inviting, not cold and corporate
- Let photos breathe with plenty of whitespace
- Have a personal, handcrafted feel
- Be simple enough that anyone can manage it

### Inspiration
The design draws from:
- **Art gallery walls** - Cream/off-white backgrounds like museum walls
- **Film photography aesthetic** - Warm, analog feel
- **Minimalist Japanese design** - Restraint, negative space, subtle details

---

## 2. Color Choices

### Why Warm Cream Instead of Pure White?

**Decision:** Background color `#f5f0eb` (warm cream) instead of `#ffffff` (white)

**Reasoning:**
- Pure white (#ffffff) feels sterile and clinical - like a hospital or Apple store
- Warm cream feels like natural paper, canvas, or gallery walls
- It's easier on the eyes for longer viewing sessions
- Creates a cozy, inviting atmosphere that matches photography's emotional nature
- Complements both warm and cool-toned photographs better than stark white

### Why Brown Accent Instead of Blue?

**Decision:** Accent color `#7a6348` (warm brown) instead of typical blue

**Reasoning:**
- Blue is overused in web design (Facebook, Twitter, LinkedIn)
- Brown/sepia tones connect to photography history (darkrooms, old prints)
- Warm brown complements the cream background harmoniously
- Feels more artistic and personal than corporate blue
- Works well with both nature and city photography themes

### Why Muted Text Colors?

**Decision:** Text colors are muted (`#2c2825`, `#5a534d`) instead of pure black

**Reasoning:**
- Pure black (#000000) on cream creates harsh contrast
- Muted dark brown feels more natural and readable
- Reduces eye strain while maintaining readability
- Keeps the focus on the colorful photographs, not the text
- Creates a cohesive, harmonious color story

---

## 3. Typography Decisions

### Why Two Fonts?

**Decision:** Cormorant Garamond (serif) + Montserrat (sans-serif)

**Reasoning:**
- **Cormorant Garamond** for personality:
  - Elegant, artistic feel for titles and navigation
  - Italic style adds a personal, handwritten quality
  - Connects to classic print design and photography books
  - The name "Thru Lens" needed a font with character

- **Montserrat** for clarity:
  - Clean, modern sans-serif for UI elements and body text
  - Easy to read at small sizes (buttons, labels)
  - Provides contrast with the decorative serif
  - Feels contemporary without being cold

### Why Light Font Weights?

**Decision:** Primarily using 300 (light) and 400 (regular) weights

**Reasoning:**
- Light weights feel delicate and elegant
- Heavy fonts would overpower the photographs
- Creates an airy, open feeling
- Matches the overall minimal aesthetic
- Light italic text feels like handwriting or journal entries

### Why Responsive Font Sizes with Clamp()?

**Decision:** Using CSS `clamp()` for fluid typography

**Reasoning:**
- The title looks good large on desktop, but would overwhelm mobile
- `clamp(1.8rem, 5vw, 2.5rem)` means:
  - Minimum: 1.8rem (never smaller)
  - Preferred: 5% of viewport width (scales with screen)
  - Maximum: 2.5rem (never larger)
- Creates smooth scaling without jarring breakpoints
- Better user experience across all devices

---

## 4. Layout Decisions

### Why a Grid Instead of Masonry?

**Decision:** CSS Grid with equal-sized squares instead of Pinterest-style masonry

**Reasoning:**
- **Consistency**: All photos get equal visual weight and importance
- **Simplicity**: Easier for viewers to scan and browse
- **Fairness**: No photo is "bigger" or more prominent than others
- **Performance**: CSS Grid is native, no JavaScript library needed
- **Mobile-friendly**: Squares reflow naturally on smaller screens

Masonry layouts can feel chaotic and make some photos feel less important based on their aspect ratio.

### Why Square Aspect Ratio?

**Decision:** All photo frames are 1:1 (square)

**Reasoning:**
- Creates visual consistency regardless of original photo dimensions
- Instagram-influenced familiarity (users understand square grids)
- Works well for both portrait and landscape photos via `object-fit: cover`
- Easier to create balanced layouts
- Square crops often focus on the most interesting part of an image

### Why Sticky Navigation?

**Decision:** Navigation bar sticks to top when scrolling

**Reasoning:**
- Users can always access navigation without scrolling back up
- Helps orientation - always know you're on the photography page
- The blur effect (`backdrop-filter: blur`) keeps it subtle
- Common pattern users expect on modern websites

### Why Sections Instead of Separate Pages?

**Decision:** Nature and City galleries on same page, scrollable

**Reasoning:**
- Single-page feels more like browsing a physical gallery
- No page load delays between sections
- Users can see everything at once, scroll naturally
- Simpler architecture, easier to maintain
- Anchor links (#nature, #city) still allow direct linking

---

## 5. Interactive Design Decisions

### Why a Single Menu Button (⋯) Instead of Multiple Buttons?

**Decision:** Consolidated all actions (Edit, Replace, Rearrange, Delete) under one menu

**Reasoning:**
- **Cleaner appearance**: One button is less cluttered than four
- **Progressive disclosure**: Actions are hidden until needed
- **Familiar pattern**: Users know the "..." menu from many apps
- **Less intimidating**: Seeing a delete button on every photo feels aggressive
- **Touch-friendly**: One larger button is easier to tap on mobile

### Why Hover Effects?

**Decision:** Photos lift up and show captions on hover

**Reasoning:**
- Creates sense of interactivity and life
- Indicates photos are clickable
- Caption appears only when needed (not cluttering the grid)
- The lift effect (`translateY(-5px)`) creates depth, like picking up a print
- Subtle zoom on the image adds visual interest

### Why a Lightbox Instead of New Page?

**Decision:** Clicking a photo opens a dark overlay, not a new page

**Reasoning:**
- Faster - no page load, instant viewing
- Context preserved - user knows they're still in the gallery
- Dark background makes photos pop (like a dark room or theater)
- Easy to browse with arrow keys
- Natural "escape to close" behavior

### Why Drag-and-Drop for Reordering?

**Decision:** Drag photos to rearrange instead of numbered inputs

**Reasoning:**
- **Intuitive**: Matches how you'd arrange physical prints on a table
- **Visual feedback**: You see the result as you work
- **Fun**: More engaging than typing numbers
- **Efficient**: Faster for making multiple changes
- **Familiar**: Users know drag-and-drop from file managers, Trello, etc.

### Why Password Protection on Actions?

**Decision:** All editing actions require password

**Reasoning:**
- Prevents accidental changes by visitors
- Simple security without complex login systems
- Password entered once per action is low friction
- Visitors can view freely, only owner can modify
- Appropriate level of security for a personal portfolio

---

## 6. Photo Management Features

### Why "Replace" Instead of Just Delete + Upload?

**Decision:** Added dedicated "Replace Photo" feature

**Reasoning:**
- Common use case: "I have a better version of this photo"
- Preserves the title and position in the gallery
- Fewer steps than delete + re-upload + re-title + re-position
- Reduces chance of mistakes
- Shows side-by-side preview (old → new) for confidence

### Why Keep Title Separate from Filename?

**Decision:** Photos have editable titles stored in metadata, not filenames

**Reasoning:**
- Filenames are ugly: `IMG_2897_1765224805794_55cefca7.jpeg`
- Titles can have spaces, special characters, proper capitalization
- Title can be changed without renaming the file
- Fallback: If no title set, we clean up the filename automatically
- Metadata stored in JSON is flexible and easy to extend

### Why Sort Order in Metadata?

**Decision:** Photo order stored as `sortOrder` number in metadata.json

**Reasoning:**
- File system order is unpredictable
- Upload date order might not match artistic preference
- Numeric order allows precise positioning
- Easy to reorder without renaming files
- Preserved across server restarts

---

## 7. Technical Decisions

### Why Vanilla JavaScript Instead of React/Vue?

**Decision:** Plain HTML, CSS, and JavaScript with no framework

**Reasoning:**
- **Simplicity**: Easier to understand and modify
- **Performance**: No framework overhead, fast page loads
- **Learning**: Better for understanding web fundamentals
- **Longevity**: Won't break when frameworks update
- **Hosting**: Static files work anywhere, no build step needed
- **Appropriate scale**: A portfolio doesn't need component architecture

### Why Node.js for the API?

**Decision:** Simple Node.js server instead of PHP, Python, or others

**Reasoning:**
- JavaScript on both frontend and backend (one language to learn)
- Native JSON handling matches the API responses
- Lightweight and fast for simple file operations
- Easy to understand the code
- Docker container keeps it isolated and reproducible

### Why Docker for the API Server?

**Decision:** API runs in Docker container, not directly on server

**Reasoning:**
- **Isolation**: API can't accidentally affect other services
- **Consistency**: Same environment everywhere
- **Easy updates**: Rebuild container with new code
- **Resource limits**: Container can be restricted if needed
- **Modern practice**: Good pattern to learn

### Why Mounted Volume for server.js?

**Decision:** server.js mounted into container instead of baked in

**Reasoning:**
- Code changes take effect with just `docker restart`
- No need to rebuild the entire container for small fixes
- Faster development iteration
- Can edit the file directly on the server
- Original Dockerfile still works for fresh deployments

---

## 8. User Experience Decisions

### Why No Login System?

**Decision:** Simple password instead of user accounts

**Reasoning:**
- Only one user (Rosy) needs to edit
- No need for registration, email verification, password reset
- Password in environment variable is simple and secure enough
- Reduces complexity significantly
- Visitors get friction-free viewing experience

### Why Confirmation on Delete?

**Decision:** Delete requires seeing the photo + typing password

**Reasoning:**
- Prevents accidental deletion
- Seeing the photo makes you think twice
- Password requirement adds intentionality
- No "undo" exists, so confirmation is essential
- Red styling signals danger/warning

### Why Auto-Format Filenames to Titles?

**Decision:** If no title set, convert filename like `sunset_beach.jpg` → "Sunset Beach"

**Reasoning:**
- Better than showing raw filenames
- Works as reasonable default
- Encourages setting real titles (seeing auto-title might prompt "I should name this properly")
- Strips timestamp and hash suffixes we add for uniqueness
- Capitalizes words for readability

---

## 9. Visual Feedback Philosophy

### Principle: Every Action Should Have Visible Response

| Action | Feedback |
|--------|----------|
| Hover on photo | Photo lifts, caption appears |
| Click menu button | Button rotates, dropdown slides in |
| Drag photo | Original fades, drop target highlights |
| Click modal button | Button shows "Replacing..." text |
| Successful action | Modal closes, gallery refreshes |
| Error | Alert with specific message |

**Why this matters:**
- Users know their action was received
- Reduces anxiety ("Did it work?")
- Creates polished, professional feel
- Helps diagnose problems ("It said invalid password")

---

## 10. What We Intentionally Left Out

### No Comments Section
**Reasoning:** This is a portfolio, not a social platform. Comments add complexity and moderation burden.

### No Like/Favorite Buttons
**Reasoning:** Portfolio should feel curated by the artist, not ranked by popularity.

### No Social Share Buttons
**Reasoning:** Can be added later if needed, but they add visual clutter. Users can copy URLs.

### No Analytics Dashboard
**Reasoning:** Keep it simple. Google Analytics could be added via script tag if wanted.

### No Image Editing
**Reasoning:** Photo editing should happen before upload in proper software (Lightroom, etc.)

### No Bulk Upload
**Reasoning:** Encourages thoughtful curation. Upload one at a time, give it a title, position it.

---

## 11. Future Considerations

If the site grows, these additions would follow the same design philosophy:

1. **Dark Mode** - Same warm feeling, just inverted (dark brown background, cream text)
2. **More Categories** - Same grid layout, just more sections
3. **Search** - Simple text input, minimal styling
4. **EXIF Display** - Small text under photos in lightbox, not prominent

---

## Conclusion

Every design decision aimed to answer: **"Does this help the photos shine?"**

If something made the site flashier but distracted from the photography, we left it out. If something was complex but genuinely useful (like drag-and-drop reordering), we included it.

The result is a website that feels personal, warm, and gallery-like - a digital home for photography that respects both the art and the viewer.

---

*Document created: December 2024*
*For: Thru My Lens - https://rosy.shitchell.com/photography/*
