import React from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Image,
} from 'react-native';
import { useTheme } from '../theme/ThemeContext';
import { usePlayerStore } from '../store/playerStore';
import { usePlayerControls, formatTime } from '../hooks/usePlayer';

interface MiniPlayerProps {
  onPress: () => void;
}

export function MiniPlayer({ onPress }: MiniPlayerProps) {
  const { colors } = useTheme();
  const { currentTrack, isPlaying, position, duration } = usePlayerStore();
  const { play, pause, skipToNext } = usePlayerControls();

  if (!currentTrack) return null;

  const progress = duration > 0 ? (position / duration) * 100 : 0;

  return (
    <TouchableOpacity
      style={[styles.container, { backgroundColor: colors.miniPlayer }]}
      onPress={onPress}
      activeOpacity={0.9}
    >
      {/* Progress bar */}
      <View style={[styles.progressBar, { backgroundColor: colors.border }]}>
        <View
          style={[
            styles.progressFill,
            { width: `${progress}%`, backgroundColor: colors.primary },
          ]}
        />
      </View>

      <View style={styles.content}>
        {/* Album art */}
        <View style={[styles.artwork, { backgroundColor: colors.surfaceVariant }]}>
          {currentTrack.artwork ? (
            <Image source={{ uri: currentTrack.artwork }} style={styles.artworkImage} />
          ) : (
            <Text style={[styles.artworkPlaceholder, { color: colors.textMuted }]}>
              {'\u266B'}
            </Text>
          )}
        </View>

        {/* Track info */}
        <View style={styles.info}>
          <Text
            style={[styles.title, { color: colors.text }]}
            numberOfLines={1}
          >
            {currentTrack.title}
          </Text>
          <Text
            style={[styles.artist, { color: colors.textSecondary }]}
            numberOfLines={1}
          >
            {currentTrack.artist || 'Unknown Artist'}
          </Text>
        </View>

        {/* Controls */}
        <View style={styles.controls}>
          <TouchableOpacity
            onPress={isPlaying ? pause : play}
            style={styles.controlButton}
          >
            <Text style={[styles.controlIcon, { color: colors.text }]}>
              {isPlaying ? '\u275A\u275A' : '\u25B6'}
            </Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={skipToNext} style={styles.controlButton}>
            <Text style={[styles.controlIcon, { color: colors.text }]}>
              {'\u23ED'}
            </Text>
          </TouchableOpacity>
        </View>
      </View>
    </TouchableOpacity>
  );
}

const styles = StyleSheet.create({
  container: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    borderTopLeftRadius: 8,
    borderTopRightRadius: 8,
    overflow: 'hidden',
  },
  progressBar: {
    height: 2,
    width: '100%',
  },
  progressFill: {
    height: '100%',
  },
  content: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: 8,
    paddingHorizontal: 12,
  },
  artwork: {
    width: 48,
    height: 48,
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    overflow: 'hidden',
  },
  artworkImage: {
    width: '100%',
    height: '100%',
  },
  artworkPlaceholder: {
    fontSize: 24,
  },
  info: {
    flex: 1,
    marginLeft: 12,
  },
  title: {
    fontSize: 14,
    fontWeight: '600',
  },
  artist: {
    fontSize: 12,
    marginTop: 2,
  },
  controls: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  controlButton: {
    padding: 8,
    marginLeft: 4,
  },
  controlIcon: {
    fontSize: 20,
  },
});
