import React from 'react';
import {
  View,
  Text,
  StyleSheet,
  Image,
  TouchableOpacity,
  Dimensions,
} from 'react-native';
import { SafeAreaView } from 'react-native-safe-area-context';
import { useTheme } from '../theme/ThemeContext';
import { usePlayerStore } from '../store/playerStore';
import { PlayerControls } from '../components/PlayerControls';
import { ProgressBar } from '../components/ProgressBar';

const { width: SCREEN_WIDTH } = Dimensions.get('window');
const ARTWORK_SIZE = SCREEN_WIDTH - 80;

export function NowPlayingScreen({ navigation }: any) {
  const { colors } = useTheme();
  const { currentTrack, queue } = usePlayerStore();

  if (!currentTrack) {
    return (
      <SafeAreaView style={[styles.container, { backgroundColor: colors.background }]}>
        <View style={styles.emptyContainer}>
          <Text style={[styles.emptyIcon, { color: colors.textMuted }]}>{'\u266B'}</Text>
          <Text style={[styles.emptyText, { color: colors.textMuted }]}>
            No track playing
          </Text>
          <Text style={[styles.emptySubtext, { color: colors.textMuted }]}>
            Select a track from your library to start playing
          </Text>
        </View>
      </SafeAreaView>
    );
  }

  return (
    <SafeAreaView style={[styles.container, { backgroundColor: colors.background }]}>
      {/* Header */}
      <View style={styles.header}>
        <TouchableOpacity onPress={() => navigation.goBack()} style={styles.backButton}>
          <Text style={[styles.backIcon, { color: colors.text }]}>{'\u2039'}</Text>
        </TouchableOpacity>
        <View style={styles.headerCenter}>
          <Text style={[styles.headerLabel, { color: colors.textSecondary }]}>
            NOW PLAYING
          </Text>
          {queue.length > 1 && (
            <Text style={[styles.queueInfo, { color: colors.textMuted }]}>
              {queue.findIndex((t) => t.id === currentTrack.id) + 1} of {queue.length}
            </Text>
          )}
        </View>
        <TouchableOpacity
          style={styles.queueButton}
          onPress={() => navigation.navigate('Queue')}
        >
          <Text style={[styles.queueIcon, { color: colors.text }]}>{'\u2630'}</Text>
        </TouchableOpacity>
      </View>

      {/* Artwork */}
      <View style={styles.artworkContainer}>
        <View
          style={[
            styles.artwork,
            { backgroundColor: colors.surfaceVariant },
          ]}
        >
          {currentTrack.artwork ? (
            <Image
              source={{ uri: currentTrack.artwork }}
              style={styles.artworkImage}
            />
          ) : (
            <Text style={[styles.artworkPlaceholder, { color: colors.textMuted }]}>
              {'\u266B'}
            </Text>
          )}
        </View>
      </View>

      {/* Track Info */}
      <View style={styles.trackInfo}>
        <Text style={[styles.title, { color: colors.text }]} numberOfLines={1}>
          {currentTrack.title}
        </Text>
        <Text style={[styles.artist, { color: colors.textSecondary }]} numberOfLines={1}>
          {currentTrack.artist || 'Unknown Artist'}
        </Text>
        {currentTrack.album && (
          <Text style={[styles.album, { color: colors.textMuted }]} numberOfLines={1}>
            {currentTrack.album}
          </Text>
        )}
      </View>

      {/* Progress Bar */}
      <View style={styles.progressContainer}>
        <ProgressBar showTime />
      </View>

      {/* Controls */}
      <View style={styles.controlsContainer}>
        <PlayerControls size="large" />
      </View>

      {/* Bottom Actions */}
      <View style={styles.bottomActions}>
        <TouchableOpacity style={styles.actionButton}>
          <Text style={[styles.actionIcon, { color: colors.textMuted }]}>{'\u2665'}</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.actionButton}>
          <Text style={[styles.actionIcon, { color: colors.textMuted }]}>{'\u2026'}</Text>
        </TouchableOpacity>
      </View>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 16,
    paddingVertical: 8,
  },
  backButton: {
    padding: 8,
    width: 44,
  },
  backIcon: {
    fontSize: 32,
    fontWeight: '300',
  },
  headerCenter: {
    alignItems: 'center',
  },
  headerLabel: {
    fontSize: 11,
    fontWeight: '600',
    letterSpacing: 1,
  },
  queueInfo: {
    fontSize: 12,
    marginTop: 2,
  },
  queueButton: {
    padding: 8,
    width: 44,
    alignItems: 'flex-end',
  },
  queueIcon: {
    fontSize: 20,
  },
  artworkContainer: {
    alignItems: 'center',
    paddingVertical: 24,
  },
  artwork: {
    width: ARTWORK_SIZE,
    height: ARTWORK_SIZE,
    borderRadius: 8,
    justifyContent: 'center',
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 8 },
    shadowOpacity: 0.3,
    shadowRadius: 12,
    elevation: 8,
  },
  artworkImage: {
    width: '100%',
    height: '100%',
    borderRadius: 8,
  },
  artworkPlaceholder: {
    fontSize: 80,
  },
  trackInfo: {
    alignItems: 'center',
    paddingHorizontal: 24,
    marginBottom: 24,
  },
  title: {
    fontSize: 22,
    fontWeight: 'bold',
    textAlign: 'center',
  },
  artist: {
    fontSize: 16,
    marginTop: 4,
    textAlign: 'center',
  },
  album: {
    fontSize: 14,
    marginTop: 4,
    textAlign: 'center',
  },
  progressContainer: {
    marginBottom: 24,
  },
  controlsContainer: {
    marginBottom: 24,
  },
  bottomActions: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    paddingHorizontal: 48,
  },
  actionButton: {
    padding: 12,
  },
  actionIcon: {
    fontSize: 24,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 48,
  },
  emptyIcon: {
    fontSize: 80,
    marginBottom: 16,
  },
  emptyText: {
    fontSize: 20,
    fontWeight: '600',
    marginBottom: 8,
  },
  emptySubtext: {
    fontSize: 15,
    textAlign: 'center',
  },
});
