import React from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
} from 'react-native';
import { SafeAreaView } from 'react-native-safe-area-context';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useTheme } from '../theme/ThemeContext';
import { ThemeMode } from '../types';

export function SettingsScreen() {
  const { colors, themeMode, setThemeMode, isDark } = useTheme();

  const handleClearData = () => {
    Alert.alert(
      'Clear All Data',
      'This will remove all your tracks and playlists. This cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Clear',
          style: 'destructive',
          onPress: async () => {
            await AsyncStorage.clear();
            Alert.alert('Done', 'All data has been cleared. Restart the app to see changes.');
          },
        },
      ]
    );
  };

  const ThemeOption = ({
    mode,
    label,
    description,
  }: {
    mode: ThemeMode;
    label: string;
    description: string;
  }) => (
    <TouchableOpacity
      style={[
        styles.option,
        { backgroundColor: colors.surface },
        themeMode === mode && { borderColor: colors.primary, borderWidth: 2 },
      ]}
      onPress={() => setThemeMode(mode)}
    >
      <View style={styles.optionContent}>
        <Text style={[styles.optionLabel, { color: colors.text }]}>{label}</Text>
        <Text style={[styles.optionDescription, { color: colors.textSecondary }]}>
          {description}
        </Text>
      </View>
      {themeMode === mode && (
        <Text style={[styles.checkmark, { color: colors.primary }]}>{'\u2713'}</Text>
      )}
    </TouchableOpacity>
  );

  return (
    <SafeAreaView style={[styles.container, { backgroundColor: colors.background }]}>
      <ScrollView>
        <Text style={[styles.title, { color: colors.text }]}>Settings</Text>

        {/* Theme Section */}
        <View style={styles.section}>
          <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
            APPEARANCE
          </Text>
          <ThemeOption
            mode="system"
            label="System"
            description="Follow device settings"
          />
          <ThemeOption mode="light" label="Light" description="Always light mode" />
          <ThemeOption mode="dark" label="Dark" description="Always dark mode" />
        </View>

        {/* About Section */}
        <View style={styles.section}>
          <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
            ABOUT
          </Text>
          <View style={[styles.infoRow, { backgroundColor: colors.surface }]}>
            <Text style={[styles.infoLabel, { color: colors.text }]}>Version</Text>
            <Text style={[styles.infoValue, { color: colors.textSecondary }]}>1.0.0</Text>
          </View>
          <View style={[styles.infoRow, { backgroundColor: colors.surface }]}>
            <Text style={[styles.infoLabel, { color: colors.text }]}>Current Theme</Text>
            <Text style={[styles.infoValue, { color: colors.textSecondary }]}>
              {isDark ? 'Dark' : 'Light'}
            </Text>
          </View>
        </View>

        {/* Data Section */}
        <View style={styles.section}>
          <Text style={[styles.sectionTitle, { color: colors.textSecondary }]}>
            DATA
          </Text>
          <TouchableOpacity
            style={[styles.dangerButton, { backgroundColor: colors.surface }]}
            onPress={handleClearData}
          >
            <Text style={[styles.dangerButtonText, { color: colors.error }]}>
              Clear All Data
            </Text>
          </TouchableOpacity>
        </View>

        {/* Footer */}
        <View style={styles.footer}>
          <Text style={[styles.footerText, { color: colors.textMuted }]}>
            Vinyl - A simple audio player
          </Text>
          <Text style={[styles.footerText, { color: colors.textMuted }]}>
            Made with {'\u2665'}
          </Text>
        </View>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  title: {
    fontSize: 28,
    fontWeight: 'bold',
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  section: {
    marginBottom: 24,
    paddingHorizontal: 16,
  },
  sectionTitle: {
    fontSize: 12,
    fontWeight: '600',
    letterSpacing: 1,
    marginBottom: 8,
    marginLeft: 4,
  },
  option: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: 16,
    borderRadius: 12,
    marginBottom: 8,
  },
  optionContent: {
    flex: 1,
  },
  optionLabel: {
    fontSize: 16,
    fontWeight: '500',
  },
  optionDescription: {
    fontSize: 13,
    marginTop: 2,
  },
  checkmark: {
    fontSize: 20,
    fontWeight: 'bold',
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 16,
    borderRadius: 12,
    marginBottom: 8,
  },
  infoLabel: {
    fontSize: 15,
  },
  infoValue: {
    fontSize: 15,
  },
  dangerButton: {
    padding: 16,
    borderRadius: 12,
    alignItems: 'center',
  },
  dangerButtonText: {
    fontSize: 15,
    fontWeight: '600',
  },
  footer: {
    alignItems: 'center',
    paddingVertical: 32,
  },
  footerText: {
    fontSize: 13,
    marginVertical: 2,
  },
});
